import express from "express";
import sqlite3 from "sqlite3";
import { open } from "sqlite";
import cors from "cors";
import dotenv from "dotenv";
import bcrypt from "bcrypt";
import jwt from "jsonwebtoken";
import rateLimit from "express-rate-limit";
import slowDown from "express-slow-down";
import helmet from "helmet";
import fetch from "node-fetch";

const app = express();
app.set("trust proxy", 1);
app.use(express.json());
app.use(helmet({ crossOriginResourcePolicy: false }));
app.use(
  cors({
    origin: ["https://seanpearce.net", "https://www.seanpearce.net"],
    credentials: false,
  })
);

dotenv.config();

const apiLimiter = rateLimit({
  windowMs: 60 * 1000, // 1 minute
  max: 60, // 60 req/min/ip
  standardHeaders: true,
  legacyHeaders: false,
});
app.use(apiLimiter);

const loginSlowdown = slowDown({
  windowMs: 15 * 60 * 1000, // 15 minutes
  delayAfter: 3, // 5 quick tries
  delayMs: () => 500, // then +500ms per
  maxDelayMs: 5000,
  validate: { delayMs: true },
});

const loginLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 min
  max: 10, // after 10, 429 error
  standardHeaders: true,
  legacyHeaders: false,
  message: { error: "Too many login attempts, try again later." },
});

const signupLimiter = rateLimit({
  windowMs: 60 * 60 * 1000, // 1 hour window
  max: 20, // up to 20 signups/IP/hour
  standardHeaders: true,
  legacyHeaders: false,
  message: { error: "Too many signups from this IP, try again later." },
});

async function ensureSessionOwner(userId, sessionId) {
  const row = await db.get(
    `SELECT 1 FROM chat_sessions WHERE id = ? AND user_id = ?`,
    [Number(sessionId), Number(userId)]
  );
  return !!row;
}

async function getOrCreateNewestSessionId(userId) {
  const newest = await db.get(
    `SELECT id FROM chat_sessions
     WHERE user_id = ?
     ORDER BY created_at DESC, id DESC
     LIMIT 1`,
    [Number(userId)]
  );
  if (newest?.id) return newest.id;

  const result = await db.run(
    `INSERT INTO chat_sessions(user_id, title) VALUES(?, ?)`,
    [Number(userId), "New chat"]
  );
  return result.lastID;
}

app.get("/ping", (req, res) => res.json({ status: "pinged" }));

async function initDb(db) {
  await db.exec(`PRAGMA foreign_keys = ON;`);

  await db.exec("BEGIN");
  try {
    await db.exec(`
      CREATE TABLE IF NOT EXISTS users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        username TEXT UNIQUE,
        password TEXT
      );
    `);

    await db.exec(`
      CREATE TABLE IF NOT EXISTS chats (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER NOT NULL,
        role TEXT NOT NULL CHECK (role IN ('user','assistant')),
        content TEXT NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id)
      );
      CREATE INDEX IF NOT EXISTS idx_chats_user_created
        ON chats(user_id, created_at);
    `);

    await db.exec(`
      CREATE TABLE IF NOT EXISTS chat_sessions (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER NOT NULL,
        title TEXT NOT NULL DEFAULT 'New chat',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id)
      );
      CREATE INDEX IF NOT EXISTS idx_chat_sessions_user_created
        ON chat_sessions(user_id, created_at);
    `);

    const cols = await db.all(`PRAGMA table_info(chats)`);
    const hasSessionId = cols.some((c) => c.name === "session_id");
    if (!hasSessionId) {
      await db.exec(`
        ALTER TABLE chats
          ADD COLUMN session_id INTEGER
          REFERENCES chat_sessions(id) ON DELETE CASCADE;
      `);
      await db.exec(`
        CREATE INDEX IF NOT EXISTS idx_chats_user_session
          ON chats(user_id, session_id, id);
      `);
    }

    const usersNeeding = await db.all(`
      SELECT DISTINCT user_id FROM chats
      WHERE session_id IS NULL OR session_id = 0
    `);

    for (const { user_id } of usersNeeding) {
      const { lastID: sessionId } = await db.run(
        `INSERT INTO chat_sessions(user_id, title) VALUES(?, ?)`,
        [user_id, "Imported chat"]
      );
      await db.run(
        `UPDATE chats SET session_id = ? WHERE user_id = ? AND (session_id IS NULL OR session_id = 0)`,
        [sessionId, user_id]
      );
    }

    await db.exec("COMMIT");
  } catch (e) {
    await db.exec("ROLLBACK");
    throw e;
  }
}

const usernameRe = /^[a-zA-Z0-9._-]{3,32}$/;

app.post("/signup", signupLimiter, async (req, res) => {
  const { username, password } = req.body;

  if (!username || !password) {
    return res.status(400).json({ error: "Username and password required" });
  }
  if (!usernameRe.test(username)) {
    return res
      .status(400)
      .json({ error: "Username must be 3–32 chars (letters, numbers, . _ -)" });
  }
  if (typeof password !== "string" || password.length < 8) {
    return res
      .status(400)
      .json({ error: "Password must be at least 8 characters" });
  }

  try {
    const hashedPassword = await bcrypt.hash(password, 10);
    await db.run("INSERT INTO users (username, password) VALUES (?, ?)", [
      username,
      hashedPassword,
    ]);
    res.json({ status: "success", username });
  } catch (err) {
    if (err.code === "SQLITE_CONSTRAINT") {
      return res.status(409).json({ error: "Username already exists" });
    }
    res.status(500).json({ error: err.message });
  }
});

async function loginHandler(req, res) {
  const { username, password } = req.body;

  try {
    const user = await db.get("SELECT * FROM users WHERE username = ?", [
      username,
    ]);

    if (!user) {
      return res.status(401).json({ error: "Invalid username or password" });
    }

    const match = await bcrypt.compare(password, user.password);
    if (!match) {
      return res.status(401).json({ error: "Invalid username or password" });
    }

    const token = jwt.sign(
      { userId: user.id, username: user.username },
      process.env.JWT_SECRET,
      { expiresIn: process.env.JWT_EXPIRES_IN }
    );

    res.json({ status: "success", token });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
}

app.post("/login", loginSlowdown, loginLimiter, loginHandler);

function authenticateJWT(req, res, next) {
  const authHeader = req.headers.authorization;

  if (!authHeader) {
    return res.sendStatus(401);
  }

  const token = authHeader.split(" ")[1];

  jwt.verify(token, process.env.JWT_SECRET, (err, user) => {
    if (err) {
      return res.status(403).json({ error: "Invalid or expired token" });
    }

    req.user = user;
    next();
  });
}

app.get("/me", authenticateJWT, async (req, res) => {
  res.json({ userId: req.user.userId, username: req.user.username });
});

app.post("/chat", authenticateJWT, async (req, res) => {
  try {
    const { messages, sessionId: rawSessionId } = req.body;

    if (!Array.isArray(messages) || messages.length === 0) {
      return res.status(400).json({ error: "Messages array required" });
    }

    let sessionId = Number(rawSessionId);
    if (!sessionId) {
      sessionId = await getOrCreateNewestSessionId(req.user.userId);
    } else if (!(await ensureSessionOwner(req.user.userId, sessionId))) {
      return res.status(404).json({ error: "Session not found" });
    }

    const last = messages[messages.length - 1];
    if (!last || last.role !== "user") {
      return res
        .status(400)
        .json({ error: "Last message must be {role:'user',content:string}" });
    }

    await db.run(
      "INSERT INTO chats (user_id, session_id, role, content) VALUES (?, ?, 'user', ?)",
      [req.user.userId, sessionId, last.content.trim()]
    );

    const history = await db.all(
      `SELECT role, content FROM chats
       WHERE user_id = ? AND session_id = ?
       ORDER BY id DESC LIMIT 20`,
      [req.user.userId, sessionId]
    );

    const past = history.reverse();

    const fullContext = [...past, last];

    const model = process.env.OLLAMA_MODEL;
    const baseUrl = process.env.OLLAMA_BASE_URL;

    const upstream = await fetch(`${baseUrl}/api/chat`, {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ model, messages: fullContext, stream: false }),
    });

    const raw = await upstream.text();
    if (!upstream.ok) {
      console.error(
        "[chat] upstream status",
        upstream.status,
        raw.slice(0, 500)
      );
      return res
        .status(502)
        .json({ error: "Ollama upstream error", body: raw.slice(0, 500) });
    }

    const data = JSON.parse(raw);
    const reply = data?.message?.content ?? data?.response ?? "";

    await db.run(
      "INSERT INTO chats (user_id, session_id, role, content) VALUES (?, ?, 'assistant', ?)",
      [req.user.userId, sessionId, reply]
    );

    res.json({ reply, sessionId });
  } catch (err) {
    console.error("Chat error:", err);
    res.status(500).json({ error: "Chat service unavailable" });
  }
});

app.get("/chats", authenticateJWT, async (req, res) => {
  let sessionId = Number(req.query.sessionId);
  if (!sessionId) {
    sessionId = await getOrCreateNewestSessionId(req.user.userId);
  } else if (!(await ensureSessionOwner(req.user.userId, sessionId))) {
    return res.status(404).json({ error: "Session not found" });
  }

  const rows = await db.all(
    "SELECT role, content, created_at FROM chats WHERE user_id = ? AND session_id = ? ORDER BY id ASC",
    [req.user.userId, sessionId]
  );
  res.json({ sessionId, messages: rows });
});

app.post("/chats/clear", authenticateJWT, async (req, res) => {
  const sessionId = Number(req.body?.sessionId);
  if (!sessionId || !(await ensureSessionOwner(req.user.userId, sessionId))) {
    return res.status(404).json({ error: "Session not found" });
  }
  await db.run("DELETE FROM chats WHERE user_id = ? AND session_id = ?", [
    req.user.userId,
    sessionId,
  ]);
  res.json({ status: "cleared" });
});

app.post("/chats/clear-all", authenticateJWT, async (req, res) => {
  await db.run("DELETE FROM chats WHERE user_id = ?", [req.user.userId]);
  res.json({ status: "cleared" });
});

app.get("/chats/sessions", authenticateJWT, async (req, res) => {
  const rows = await db.all(
    `SELECT id, title, created_at FROM chat_sessions
     WHERE user_id = ? ORDER BY created_at DESC, id DESC`,
    [req.user.userId]
  );
  res.json({ sessions: rows });
});

app.post("/chats/sessions", authenticateJWT, async (req, res) => {
  const title = (req.body?.title || "New chat").toString().slice(0, 100);
  const result = await db.run(
    `INSERT INTO chat_sessions(user_id, title) VALUES(?, ?)`,
    [req.user.userId, title || "New chat"]
  );
  const row = await db.get(
    `SELECT id, title, created_at FROM chat_sessions WHERE id = ?`,
    [result.lastID]
  );
  res.status(201).json({ session: row });
});

app.patch("/chats/sessions/:id", authenticateJWT, async (req, res) => {
  const sessionId = Number(req.params.id);
  if (!(await ensureSessionOwner(req.user.userId, sessionId)))
    return res.status(404).json({ error: "Session not found" });

  const title = (req.body?.title || "").toString().trim().slice(0, 100);
  if (!title) return res.status(400).json({ error: "Title required" });

  await db.run(`UPDATE chat_sessions SET title = ? WHERE id = ?`, [
    title,
    sessionId,
  ]);
  res.json({ status: "updated" });
});

app.delete("/chats/sessions/:id", authenticateJWT, async (req, res) => {
  const sessionId = Number(req.params.id);
  if (!(await ensureSessionOwner(req.user.userId, sessionId)))
    return res.status(404).json({ error: "Session not found" });

  await db.run(`DELETE FROM chat_sessions WHERE id = ?`, [sessionId]);
  res.json({ status: "deleted" });
});

let db;
(async () => {
  try {
    db = await open({
      filename: process.env.DB_FILE,
      driver: sqlite3.Database,
    });
    await initDb(db); 

    app.listen(process.env.PORT, () =>
      console.log("Server running on port " + process.env.PORT)
    );
  } catch (err) {
    console.error("Startup failed:", err);
    process.exit(1);
  }
})();
