(function initWhenReady() {
  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", init, { once: true });
  } else {
    init();
  }
})();

function init() {
  if (sessionStorage.getItem("token")) {
    window.location.replace("/app/");
    return;
  }

  const form = document.getElementById("signupForm");
  if (!form) return;

  const usernameInput = document.getElementById("username");
  const passwordInput = document.getElementById("password");
  const usernameError = document.getElementById("usernameError");
  const passwordError = document.getElementById("passwordError");
  const usernameGroup =
    document.getElementById("usernameGroup") ||
    usernameInput.closest(".form-group");
  const passwordGroup =
    document.getElementById("passwordGroup") ||
    passwordInput.closest(".form-group");
  const successMessage = document.getElementById("successMessage");
  const submitBtn = form.querySelector(".login-btn");
  const btnText = form.querySelector(".btn-text");
  const btnLoader = form.querySelector(".btn-loader");
  const passwordToggle = document.getElementById("passwordToggle");

  // Helpers
  const setLoading = (is) => {
    submitBtn.disabled = is;
    submitBtn.classList.toggle("loading", is);
    if (btnText) btnText.style.opacity = is ? 0 : 1;
    if (btnLoader) btnLoader.style.opacity = is ? 1 : 0;
  };

  function setFieldError(inputEl, errorEl, groupEl, msg) {
    if (msg) {
      errorEl.textContent = msg;
      errorEl.classList.add("show");
      groupEl?.classList.add("error");
      inputEl.setAttribute("aria-invalid", "true");
    } else {
      errorEl.textContent = "";
      errorEl.classList.remove("show");
      groupEl?.classList.remove("error");
      inputEl.setAttribute("aria-invalid", "false");
    }
  }

  function clearErrors() {
    setFieldError(usernameInput, usernameError, usernameGroup, "");
    setFieldError(passwordInput, passwordError, passwordGroup, "");
  }

  if (passwordToggle) {
    passwordToggle.addEventListener("click", () => {
      const type = passwordInput.type === "password" ? "text" : "password";
      passwordInput.type = type;
      passwordToggle.classList.toggle("visible", type === "text");
    });
  }

  usernameInput.addEventListener("input", () => {
    const u = usernameInput.value.trim();
    if (u && /^[a-zA-Z0-9._-]{3,32}$/.test(u)) {
      setFieldError(usernameInput, usernameError, usernameGroup, "");
    }
  });
  passwordInput.addEventListener("input", () => {
    if (passwordInput.value.length >= 8) {
      setFieldError(passwordInput, passwordError, passwordGroup, "");
    }
  });

  form.addEventListener("submit", async (e) => {
    e.preventDefault();
    clearErrors();

    const username = (usernameInput.value || "").trim();
    const password = passwordInput.value || "";

    if (!username) {
      setFieldError(
        usernameInput,
        usernameError,
        usernameGroup,
        "Username is required."
      );
      usernameInput.focus();
      return;
    }
    if (!/^[a-zA-Z0-9._-]{3,32}$/.test(username)) {
      setFieldError(
        usernameInput,
        usernameError,
        usernameGroup,
        "3–32 chars: letters, numbers, . _ -"
      );
      usernameInput.focus();
      return;
    }
    if (!password) {
      setFieldError(
        passwordInput,
        passwordError,
        passwordGroup,
        "Password is required."
      );
      passwordInput.focus();
      return;
    }
    if (password.length < 8) {
      setFieldError(
        passwordInput,
        passwordError,
        passwordGroup,
        "At least 8 characters."
      );
      passwordInput.focus();
      return;
    }

    try {
      setLoading(true);

      const sRes = await fetch("/api/signup", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ username, password }),
      });
      const sData = await sRes.json().catch(() => ({}));

      if (!sRes.ok) {
        if (sRes.status === 409) {
          setFieldError(
            usernameInput,
            usernameError,
            usernameGroup,
            "Username already exists."
          );
          usernameInput.focus();
        } else if (sRes.status === 429) {
          setFieldError(
            usernameInput,
            usernameError,
            usernameGroup,
            sData?.error || "Too many signups, try later."
          );
        } else {
          setFieldError(
            passwordInput,
            passwordError,
            passwordGroup,
            sData?.error || "Could not create account."
          );
        }
        return;
      }

      if (successMessage) successMessage.classList.add("show");
      form.style.display = "none";

      const lRes = await fetch("/api/login", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ username, password }),
      });
      const lData = await lRes.json().catch(() => ({}));

      if (!lRes.ok || !lData?.token) {
        window.location.replace("/login/");
        return;
      }

      sessionStorage.setItem("token", lData.token);
      window.location.replace("/app/");
    } catch (err) {
      console.error(err);
      setFieldError(
        passwordInput,
        passwordError,
        passwordGroup,
        "Something went wrong. Please try again."
      );
    } finally {
      setLoading(false);
    }
  });
}
