(function initWhenReady() {
  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", init, { once: true });
  } else {
    init();
  }
})();

function init() {
  if (sessionStorage.getItem("token")) {
    window.location.replace("/app/");
    return;
  }

  const form = document.getElementById("loginForm");
  if (!form) return;

  const usernameInput = document.getElementById("username");
  const passwordInput = document.getElementById("password");
  const usernameError = document.getElementById("usernameError");
  const passwordError = document.getElementById("passwordError");
  const usernameGroup =
    document.getElementById("usernameGroup") ||
    usernameInput.closest(".form-group");
  const passwordGroup =
    document.getElementById("passwordGroup") ||
    passwordInput.closest(".form-group");
  const successMessage = document.getElementById("successMessage");
  const submitBtn = form.querySelector(".login-btn");
  const btnText = form.querySelector(".btn-text");
  const btnLoader = form.querySelector(".btn-loader");
  const passwordToggle = document.getElementById("passwordToggle");

  const setLoading = (is) => {
    submitBtn.disabled = is;
    submitBtn.classList.toggle("loading", is);
    if (btnText) btnText.style.opacity = is ? 0 : 1;
    if (btnLoader) btnLoader.style.opacity = is ? 1 : 0;
  };

  function setFieldError(inputEl, errorEl, groupEl, msg) {
    if (msg) {
      errorEl.textContent = msg;
      errorEl.classList.add("show");
      groupEl?.classList.add("error");
      inputEl.setAttribute("aria-invalid", "true");
    } else {
      errorEl.textContent = "";
      errorEl.classList.remove("show");
      groupEl?.classList.remove("error");
      inputEl.setAttribute("aria-invalid", "false");
    }
  }

  function clearErrors() {
    setFieldError(usernameInput, usernameError, usernameGroup, "");
    setFieldError(passwordInput, passwordError, passwordGroup, "");
  }

  if (passwordToggle) {
    passwordToggle.addEventListener("click", () => {
      const type = passwordInput.type === "password" ? "text" : "password";
      passwordInput.type = type;
      passwordToggle.classList.toggle("visible", type === "text");
    });
  }

  usernameInput.addEventListener("input", () => {
    if (usernameInput.value.trim()) {
      setFieldError(usernameInput, usernameError, usernameGroup, "");
    }
  });
  passwordInput.addEventListener("input", () => {
    if (passwordInput.value) {
      setFieldError(passwordInput, passwordError, passwordGroup, "");
    }
  });

  form.addEventListener("submit", async (e) => {
    e.preventDefault();
    clearErrors();

    const username = (usernameInput.value || "").trim();
    const password = passwordInput.value || "";

    if (!username) {
      setFieldError(
        usernameInput,
        usernameError,
        usernameGroup,
        "Username is required."
      );
      usernameInput.focus();
      return;
    }
    if (!password) {
      setFieldError(
        passwordInput,
        passwordError,
        passwordGroup,
        "Password is required."
      );
      passwordInput.focus();
      return;
    }

    try {
      setLoading(true);
      const res = await fetch("/api/login", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ username, password }),
      });

      let data = {};
      try {
        data = await res.json();
      } catch {}

      if (!res.ok || !data?.token) {
        setFieldError(
          passwordInput,
          passwordError,
          passwordGroup,
          data?.error || "Invalid username or password."
        );
        return;
      }

      if (successMessage) successMessage.classList.add("show");
      form.style.display = "none";

      sessionStorage.setItem("token", data.token);
      window.location.replace("/app/");
    } catch (err) {
      console.error(err);
      setFieldError(
        passwordInput,
        passwordError,
        passwordGroup,
        "Something went wrong. Please try again."
      );
    } finally {
      setLoading(false);
    }
  });
}
