const token = sessionStorage.getItem("token");
if (!token) window.location.replace("/login/");

const chatLog = document.getElementById("chatLog");
const form = document.getElementById("chatForm");
const input = document.getElementById("userInput");
const logoutBtn = document.getElementById("logoutBtn");
const clearBtn = document.getElementById("clearBtn");

const sessionSelect = document.getElementById("sessionSelect");
const newSessionBtn = document.getElementById("newSessionBtn");
const renameSessionBtn = document.getElementById("renameSessionBtn");
const deleteSessionBtn = document.getElementById("deleteSessionBtn");

let history = [];
let activeSessionId = null;

function addMessage(role, text) {
  const div = document.createElement("div");
  div.className = "msg " + role;
  div.textContent = (role === "user" ? "🧑 " : "🤖 ") + text;
  chatLog.appendChild(div);
  chatLog.scrollTop = chatLog.scrollHeight;
}

async function fetchJSON(url, opts = {}) {
  const res = await fetch(url, {
    ...opts,
    headers: {
      "Content-Type": "application/json",
      Authorization: "Bearer " + (sessionStorage.getItem("token") || ""),
      ...(opts.headers || {}),
    },
  });

  if (res.status === 401 || res.status === 403) {
    sessionStorage.removeItem("token");
    window.location.replace("/login/");
    return { ok: false, status: res.status, data: { error: "unauthorized" } };
  }

  const raw = await res.text();
  try {
    return { ok: res.ok, status: res.status, data: JSON.parse(raw) };
  } catch {
    return { ok: res.ok, status: res.status, data: raw };
  }
}

async function loadSessions() {
  const { ok, data } = await fetchJSON("/api/chats/sessions");
  if (!ok) return console.error("Failed to load sessions:", data);
  const sessions = data.sessions || [];
  sessionSelect.innerHTML = "";
  for (const s of sessions) {
    const opt = document.createElement("option");
    opt.value = s.id;
    opt.textContent = `${s.title} • ${new Date(s.created_at).toLocaleString()}`;
    sessionSelect.appendChild(opt);
  }
  if (sessions.length) {
    activeSessionId = activeSessionId || sessions[0].id;
    sessionSelect.value = String(activeSessionId);
  } else {
    const created = await createSession("New chat");
    activeSessionId = created.id;
    await loadSessions();
  }
}

async function loadHistory() {
  const { ok, data } = await fetchJSON(
    `/api/chats?sessionId=${activeSessionId}`
  );
  if (!ok) return console.error("Failed to load history:", data);
  history = data.messages || [];
  chatLog.innerHTML = "";
  for (const msg of history) addMessage(msg.role, msg.content);
}

async function createSession(title) {
  const { ok, data } = await fetchJSON("/api/chats/sessions", {
    method: "POST",
    body: JSON.stringify({ title }),
  });
  if (!ok) throw new Error(data.error || "Failed to create session");
  return data.session;
}

async function renameSession(id, title) {
  const { ok, data } = await fetchJSON(`/api/chats/sessions/${id}`, {
    method: "PATCH",
    body: JSON.stringify({ title }),
  });
  if (!ok) throw new Error(data.error || "Failed to rename");
  return true;
}

async function deleteSession(id) {
  const { ok, data } = await fetchJSON(`/api/chats/sessions/${id}`, {
    method: "DELETE",
  });
  if (!ok) throw new Error(data.error || "Failed to delete");
  return true;
}

form.addEventListener("submit", async (e) => {
  e.preventDefault();
  const userMsg = input.value.trim();
  if (!userMsg) return;

  addMessage("user", userMsg);
  input.value = "";

  try {
    const { ok, status, data } = await fetchJSON("/api/chat", {
      method: "POST",
      body: JSON.stringify({
        sessionId: activeSessionId,
        messages: [...history, { role: "user", content: userMsg }],
      }),
    });

    if (!ok) {
      addMessage(
        "assistant",
        `⚠️ Error ${status}: ${
          typeof data === "string" ? data : JSON.stringify(data)
        }`
      );
      return;
    }

    const reply =
      data.reply ||
      data.message?.content ||
      data.output?.message?.content ||
      "";

    if (!reply) {
      addMessage("assistant", "⚠️ No reply in response.");
      return;
    }

    addMessage("assistant", reply);
    history.push({ role: "user", content: userMsg });
    history.push({ role: "assistant", content: reply });

    if (data.sessionId && data.sessionId !== activeSessionId) {
      activeSessionId = data.sessionId;
      await loadSessions();
    }
  } catch (err) {
    console.error(err);
    addMessage("assistant", "⚠️ Network error talking to chat service.");
  }
});

sessionSelect.addEventListener("change", async () => {
  activeSessionId = Number(sessionSelect.value);
  await loadHistory();
});

newSessionBtn.addEventListener("click", async () => {
  const title = prompt("Name for the new chat:", "New chat");
  const session = await createSession(title || "New chat");
  activeSessionId = session.id;
  await loadSessions();
  await loadHistory();
});

renameSessionBtn.addEventListener("click", async () => {
  const current =
    sessionSelect.options[sessionSelect.selectedIndex]?.textContent || "";
  const title = prompt("Rename chat to:", current.split(" • ")[0]);
  if (!title) return;
  await renameSession(activeSessionId, title);
  await loadSessions();
});

deleteSessionBtn.addEventListener("click", async () => {
  if (!confirm("Delete this chat and all its messages? This cannot be undone."))
    return;
  await deleteSession(activeSessionId);
  await loadSessions();
  activeSessionId = Number(sessionSelect.value);
  await loadHistory();
});

clearBtn.addEventListener("click", async () => {
  if (!confirm("Clear all messages in this chat?")) return;
  await fetchJSON("/api/chats/clear", {
    method: "POST",
    body: JSON.stringify({ sessionId: activeSessionId }),
  });
  chatLog.innerHTML = "";
  history = [];
});

logoutBtn.addEventListener("click", () => {
  sessionStorage.removeItem("token");
  window.location.replace("/login/");
});

async function loadUser() {
  try {
    const res = await fetch("/api/me", {
      headers: { Authorization: "Bearer " + token },
    });
    if (!res.ok) throw new Error("Auth failed");
    const user = await res.json();
    const title = document.getElementById("chatTitle");
    if (user.username && title) title.textContent = `${user.username}'s Chat`;
  } catch (err) {
    sessionStorage.removeItem("token");
    window.location.replace("/login/");
  }
}

await loadUser();
await loadSessions();
await loadHistory();
